/*****************************************************************************
**+------------------------------------------------------------------------+**
**|                                                                        |**
**|                Copyright 2010 Mistral Solutions Pvt Ltd.               |**
**|                                                                        |**
**|                                                                        |**
**|                                                                        |**   
**| This program is free software; you can redistribute it and/or          |**
**| modify it under the terms of the GNU General Public License as         |**
**| published by the Free Software Foundation; either version 2 of         |**
**| the License, or (at your option) any later version.                    |**
**|                                                                        |**
**| This program is distributed in the hope that it will be useful,        |**
**| but WITHOUT ANY WARRANTY; without even the implied warranty of         |**
**| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the           |**
**| GNU General Public License for more details.                           |**
**|                                                                        |**      
**| You should have received a copy of the GNU General Public License      |**
**| along with this program; if not, write to the Free Software            |**
**| Foundation, Inc., 59 Temple Place, Suite 330, Boston,                  |**
**| MA 02111-1307 USA                                                      |**
**+------------------------------------------------------------------------+**
*****************************************************************************/ 

/**
 * \file    nandflash.c
 *
 * \brief   Routines to initialize & Access the NAND flash
 *
 * This file contains the routines for configuring and
 * accessing the NAND Flash on the DM814X EVM.
 *
 * \author  0043
 *
 * \version 01a,25aug,2010 Created
 */

/*
 *====================
 * Includes
 *====================
 */
#include "DM814x_gpmc.h"
#include "DM814x_types.h"
#include "DM814x_EVM.h"
#include "stdio.h"

#include "nandflash.h"
 
/*
 *====================
 * Defines
 *====================
 */
 #define NAND_MANF_ID	0x2C
/*
 *====================
 * Function implementations
 *====================
 */

/** **************************************************************************
 * \n \brief Routine to initialize the NAND Flash 
 *
 * This routine initializes the NAND flash by configuring GPMC .Reads the
 * Manufacturer Id ,Device Id and verifies the manufacture Id.
 *  
 *
 * \return
 * \n      return SUCCESS for success  
 * \n      return FAILED for error   
 * 
 */
STATUS nand_init(void)
{
	STATUS u32RetVal = SUCCESS;
    UINT16 mfgid, devid;
	UINT16 u16Buff[4];

    /*------------------------------------------------------------------*
     *                                                                  *
     *  NAND Flash timing parameters                                    *
     *                                                                  *
     *  GPMC.CLK freq   =                      							*
     *  GPMC.CLK period =                           					*
     *                                                                  *
     *------------------------------------------------------------------*/
     
    /* Configure GPMC */
    GPMC_SYSCONFIG = 0x0;
    GPMC_IRQENABLE = 0x0;
    GPMC_TIMEOUT_CONTROL = 0x0;
    GPMC_CONFIG = 0x10;

    GPMC_CONFIG1_0 = 0x00001800;
    GPMC_CONFIG2_0 = 0x00141400;
    GPMC_CONFIG3_0 = 0x00141400;
    GPMC_CONFIG4_0 = 0x0F010F01;
    GPMC_CONFIG5_0 = 0x010C1414;
    GPMC_CONFIG6_0 = 0x1f0f0A80;
    GPMC_CONFIG7_0 = 0x00000C44;

    NANDFLASH_CLE = CMD_RESET;
    DM814x_usecDelay(100000);

    // Start erase operation
    NANDFLASH_CLE = CMD_READID;
    NANDFLASH_ALE = 0x00;

    // Wait until command completion
    DM814x_usecDelay(10);	//TODO

    // Get IDs
    mfgid = NANDFLASH_DATA;
    devid = NANDFLASH_DATA;

    printf("NAND MFGID = %02x\n", mfgid);
    printf("NAND DEVID = %02x\n", devid);
    
    if(mfgid != NAND_MANF_ID)
    {
    	printf ("Unsupported Manf ID.\r\n");
    	u32RetVal = FAILED;
    }

    // Start erase operation
    NANDFLASH_CLE = CMD_READID;
    NANDFLASH_ALE = 0x20;
	u16Buff[0] = NANDFLASH_DATA;
	u16Buff[1] = NANDFLASH_DATA;
	u16Buff[2] = NANDFLASH_DATA;
	u16Buff[3] = NANDFLASH_DATA;
	

    // Wait until command completion
    DM814x_usecDelay(10);	//TODO

    
    return u32RetVal;
}
/** **************************************************************************
 * \n \brief Routine to Erase the Nand Flash 
 *
 * This routine Erases a block of Nand Flash .This function takes Block num as 
 * input parameter.Reads erase status and checks for successfull erase operation.
 * 
 *
 * \param blocknum 		[IN]		Block to be erased
 * 
 *
 * \return
 * \n      return SUCCESS for success  
 * \n      return FAILED for error   
 * 
 */
STATUS nand_eraseBlock(UINT32 blocknum)
{
	STATUS u32RetVal = SUCCESS;
    UINT32 status;

    // Start erase operation
    NANDFLASH_CLE = CMD_ERASE;
    NANDFLASH_ALE = (blocknum & 0x0003) << 6;
    NANDFLASH_ALE = (blocknum & 0x03fc) >> 2;
    NANDFLASH_ALE = (blocknum & 0x0c00) >> 10;
    NANDFLASH_CLE = CMD_ERASE_CONFIRM;

    // Wait for erase completion
    NANDFLASH_CLE = CMD_STATUS;
    DM814x_usecDelay(25);	//TODO

    while ((NANDFLASH_DATA & 0x20) == 0);

    // Check erase status
    NANDFLASH_CLE = CMD_STATUS;
    status = NANDFLASH_DATA;

    NANDFLASH_CLE = CMD_RESET;
    
    if (status & 0x01)
    {
    	printf ("NAND Errase Failed.\r\n");
    	u32RetVal = FAILED;
    }
    else
    {
    	printf ("NAND Errase successful.\r\n");
    }
    
	return u32RetVal;  // OK
}
/** **************************************************************************
 * \n \brief Routine to read from  Nand Flash 
 *
 * This routine Reads a page from Nand Flash .This function takes Page number 
 * as input parameter along with read buffer pointer and length of buffer.
 * 
 *
 * \param pagenum 		[IN]		Pagenumber to read from
 * \param *buffer       [IN]        Buffer pointer to copy read data
 * \param len           [IN]        Length of the buffer
 *
 * 
 * \return
 * \n      return SUCCESS for success  
 * \n      return FAILED for error   
 * 
 */

STATUS nand_readPage(UINT32 pagenum, UINT16 *buf, UINT16 len)
{
	STATUS u32RetVal = SUCCESS;
    UINT16 *dptr;
    UINT16 i;

    NANDFLASH_CLE = CMD_READ;
    NANDFLASH_ALE = 0x00;
    NANDFLASH_ALE = 0x00;
    NANDFLASH_ALE = pagenum & 0xff;
    NANDFLASH_ALE = (pagenum & 0x00ff00) >> 8;
    NANDFLASH_ALE = (pagenum & 0x030000) >> 16;
    NANDFLASH_CLE = CMD_READ_CONFIRM;

    // Wait for read completion
    NANDFLASH_CLE = CMD_STATUS;
    while ((NANDFLASH_DATA & 0x20) == 0);

    // Read data
    NANDFLASH_CLE = CMD_READ;
    dptr = buf;
    for (i = 0; i < len/2; i++)
        *dptr++ = NANDFLASH_DATA;

    return u32RetVal;
}
/** **************************************************************************
 * \n \brief Routine to Write into  Nand Flash 
 *
 * This routine Writes a page to  Nand Flash .This function takes Page number 
 * as input parameter along with  buffer pointer and length of buffer.
 * 
 *
 * \param pagenum 		[IN]       Page to be written 
 * \param *buffer       [IN]       Buffer pointer which contains data to be written 
 * \param len           [IN]       Length of the buffer
 *
 * 
 * \return
 * \n      return SUCCESS for success  
 * \n      return FAILED for error   
 * 
 */
STATUS nand_writePage(UINT32 pagenum, UINT16 *buf, UINT16 len)
{
	STATUS u32RetVal = SUCCESS;
    UINT16 *dptr;
    UINT16 i;
    UINT32 status;

    // Start program operation
    NANDFLASH_CLE = CMD_PROGRAM;
    NANDFLASH_ALE = 0x00;
    NANDFLASH_ALE = 0x00;
    NANDFLASH_ALE = pagenum & 0xff;
    NANDFLASH_ALE = (pagenum & 0x00ff00) >> 8;
    NANDFLASH_ALE = (pagenum & 0x030000) >> 16;

    dptr = buf;
    for (i = 0; i < len/2; i++)
        NANDFLASH_DATA = *dptr++;
    NANDFLASH_CLE = CMD_PROGRAM_CONFIRM;

    DM814x_usecDelay(25);	//TODO

    // Wait for program completion
    NANDFLASH_CLE = CMD_STATUS;
    while ((NANDFLASH_DATA & 0x20) == 0);

    // Check program status
    NANDFLASH_CLE = CMD_STATUS;
    status = NANDFLASH_DATA;

    if (status & 0x01)
    {
    	u32RetVal = FAILED;
    }
    
    return u32RetVal;  // OK
}
